/**
 * md2ld.js (Microdata to Linked Data)
 * 
 * Microdata から Linked Data (JSON-LD) を生成し head要素 に script[type="application/ld+json"] を追加します
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * @package md2ld.js
 * @since 1.0.0
 * @author Shimotsuki
 * @copyright Copyright (C) 2025 Shimotsuki
 * @license http://www.gnu.org/licenses/gpl-2.0.html GPL v2 or later
 * @link https://shimotsuki.wwwxyz.jp/
 */
new class {

	/**
	 * @type {number} ユニークID 生成のためのカウンター
	 */
	idCounter = 1;

	constructor() {
		const jsonLd = this.generate();

		if ( jsonLd ) {
			const script = document.createElement( 'script' );
			script.type = 'application/ld+json';
			script.textContent = jsonLd;
			document.head.append( script );

			document.querySelectorAll( '[itemscope], [itemtype], [itemid], [itemref], [itemprop]' ).forEach( ( element ) => {
				element.removeAttribute( 'itemscope' );
				element.removeAttribute( 'itemtype' );
				element.removeAttribute( 'itemid' );
				element.removeAttribute( 'itemref' );
				element.removeAttribute( 'itemprop' );
				element.removeAttribute( 'data-ld-id' ); // getEntityId() で使用
			} );
			document.querySelectorAll( '[itemprop]:is(link, meta)' ).forEach( ( element ) => {
				element.remove();
			} );
		}
	}

	/**
	 * Microdata を DOM から抽出し JSON-LD文字列 を生成
	 * 
	 * @returns {string|null}
	 */
	generate() {
		const final_graph = {};

		// 1. すべてのトップレベルエンティティを処理し final_graph を構築
		document.querySelectorAll( '[itemscope][itemtype]:not([itemprop])' ).forEach( ( entity_node ) => {
			this.processEntity( entity_node, final_graph );
		} );

		if ( 0 === Object.keys( final_graph ).length ) {
			return null;
		}

		// 2. final_graph を JSON-LD形式 に変換
		const jsonLdObject = {
			"@context": "https://schema.org",
			"@graph": this.buildJsonLdGraph( final_graph )
		};

		return JSON.stringify( jsonLdObject, null, 2 );
	}

	/**
	 * エンティティを処理し final_graph に追加
	 * ネストされたエンティティの処理にも使用
	 * 
	 * @param   {Element} entity_node - itemscope を持つノード
	 * @param   {object}  final_graph - 収集された全エンティティのマップ
	 * @returns {string}              - エンティティID
	 */
	processEntity( entity_node, final_graph ) {
		const entity_id = this.getEntityId( entity_node );

		// 既に処理済みの場合はスキップ
		if ( final_graph[ entity_id ] ) {
			return entity_id;
		}

		final_graph[ entity_id ] = {
			'id': entity_id,
			'type': entity_node.getAttribute( 'itemtype' ),
			// ネストされたエンティティも再帰的に処理
			'properties': this.collectProperties( entity_node, final_graph ),
		};

		return entity_id;
	}

	/**
	 * エンティティID を取得または生成
	 * 
	 * @param   {Element} node - itemscopeを持つノード
	 * @returns {string}       - エンティティID
	 */
	getEntityId( node ) {
		if ( node.hasAttribute( 'itemid' ) ) {
			return node.getAttribute( 'itemid' );
		}

		// ユニークID を生成
		let id = node.dataset.ldId;
		if ( ! id ) {
			id = '#ld' + String( this.idCounter++ ).padStart( 2, '0' );
			node.dataset.ldId = id;
		}
		return id;
	}

	/**
	 * エンティティの全プロパティを収集
	 * 
	 * @param   {Element} entity_node - itemscope を持つノード
	 * @param   {object}  final_graph - 収集された全エンティティのマップ
	 * @returns {object}              - プロパティを格納したオブジェクト
	 */
	collectProperties( entity_node, final_graph ) {
		const properties = {};

		// entity_node の子ノードから走査を開始
		this.recursivelyCollectProperties( entity_node.children, entity_node, final_graph, properties );

		return properties;
	}

	/**
	 * エンティティノードの子孫を再帰的に走査し、プロパティを収集するヘルパー関数
	 * 
	 * @param {HTMLCollection} current_nodes - 現在処理中のノードのコレクション
	 * @param {Element}        entity_root   - 収集対象のエンティティのルートノード
	 * @param {object}         final_graph   - 全エンティティのマップ
	 * @param {object}         properties    - 収集したプロパティの格納先
	 */
	recursivelyCollectProperties( current_nodes, entity_root, final_graph, properties ) {
		for ( const node of current_nodes ) {
			if ( node.nodeType !== Node.ELEMENT_NODE ) {
				continue;
			}

			const has_itemprop = node.hasAttribute( 'itemprop' );
			const has_itemscope = node.hasAttribute( 'itemscope' );

			// 1. itemprop を持つ要素
			if ( has_itemprop ) {
				// A. ネストされたエンティティのルートの場合 (itemprop + itemscope)
				if ( has_itemscope ) {
					const prop_names = node.getAttribute( 'itemprop' ).split( /\s+/ ).filter( name => name );
					const prop_value_data = this.getPropertyValue( node, final_graph );

					if ( null !== prop_value_data ) {
						prop_names.forEach( ( name ) => {
							properties[ name ] = properties[ name ] || [];
							properties[ name ].push( prop_value_data );
						} );
					}

					continue;
				}

				// B. 純粋なプロパティの場合 (itemprop)
				else {
					const prop_names = node.getAttribute( 'itemprop' ).split( /\s+/ ).filter( name => name );
					const prop_value_data = this.getPropertyValue( node, final_graph );

					if ( null !== prop_value_data ) {
						prop_names.forEach( ( name ) => {
							properties[ name ] = properties[ name ] || [];
							properties[ name ].push( prop_value_data );
						} );
					}

					// 子孫に itemprop がないか探す
					this.recursivelyCollectProperties( node.children, entity_root, final_graph, properties );

					continue;
				}
			}

			// 2. itemprop を持たず itemscope を持つ要素
			if ( has_itemscope ) {
				continue;
			}

			// 3. どちらの属性も持たない一般的な要素
			// 子孫に itemprop がないか探す
			this.recursivelyCollectProperties( node.children, entity_root, final_graph, properties );
		} // for
	}

	/**
	 * Microdata のルールに従いプロパティの値を確定
	 * 
	 * @param   {Element} node        - itemprop を持つノード
	 * @param   {object}  final_graph - 全エンティティのマップ
	 * @returns {object|null}
	 */
	getPropertyValue( node, final_graph ) {
		const tag = node.tagName.toLowerCase();

		// 1. itemscope がある場合
		if ( node.hasAttribute( 'itemscope' ) ) {
			// ネストされたエンティティを再帰的に処理し ID を取得
			const nested_entity_id = this.processEntity( node, final_graph );
			return {
				'value': nested_entity_id,
				'is_reference': true
			};
		}

		// 2. 属性から値を取得
		// https://html.spec.whatwg.org/#values
		let value = null;
		if ( 'meta' === tag ) {
			value = node.getAttribute( 'content' );
		} else if ( 'audio' === tag || 'embed' === tag || 'iframe' === tag || 'img' === tag || 'source' === tag || 'track' === tag || 'video' === tag ) {
			value = node.getAttribute( 'src' );
		} else if ( 'a' === tag || 'area' === tag || 'link' === tag ) {
			value = node.getAttribute( 'href' );
		} else if ( 'object' === tag ) {
			value = node.getAttribute( 'data' );
		} else if ( 'data' === tag || 'meter' === tag ) {
			value = node.getAttribute( 'value' );
		} else if ( 'time' === tag ) {
			value = node.getAttribute( 'datetime' );
		}
		if ( null !== value ) {
			return {
				'value': value,
				'is_reference': false
			};
		}

		// 3. テキストコンテンツを取得
		const text_content = this.collectTextContent( node );
		if ( text_content ) {
			return {
				'value': text_content,
				'is_reference': false
			};
		}

		return null;
	}

	/**
	 * Microdata のテキストコンテンツ結合ルールに従いノードのテキストコンテンツを収集
	 * 
	 * @param   {Element} node - itemprop を持つノード
	 * @returns {string}       - テキストコンテンツ
	 */
	collectTextContent( node ) {
		let parts = [];

		for ( const child_node of node.childNodes ) {
			if ( child_node.nodeType === Node.TEXT_NODE ) {
				const trimmed = child_node.textContent.trim();
				if ( 0 < trimmed.length ) {
					parts.push( trimmed );
				}
				continue;
			}

			if ( child_node.nodeType === Node.ELEMENT_NODE ) {
				if ( ! child_node.hasAttribute( 'itemscope' ) ) {
					continue;
				}

				if ( 'img' === child_node.tagName.toLowerCase() ) {
					const alt = child_node.getAttribute( 'alt' ) || '';
					if ( alt ) {
						parts.push( alt );
					}
					continue;
				}

				const inner_text = this.collectTextContent( child_node );
				if ( inner_text ) {
					parts.push( inner_text );
				}
			}
		} // for

		return parts.join( ' ' ).replace( /\s+/g, ' ' ).trim();
	}

	/**
	 * Microdata のテキストコンテンツ結合ルールに従いノードのテキストコンテンツを収集
	 * 
	 * @param   {Element} node - itemprop を持つノード
	 * @returns {string}       - テキストコンテンツ
	 */
	collectTextContent( node ) {
		let parts = [];

		for ( const child_node of node.childNodes ) {
			if ( child_node.nodeType === Node.TEXT_NODE ) {
				const trimmed = child_node.textContent.trim();
				if ( 0 < trimmed.length ) {
					parts.push( trimmed );
				}
				continue;
			}

			if ( child_node.nodeType === Node.ELEMENT_NODE ) {
				if ( 'img' === child_node.tagName.toLowerCase() ) {
					const alt = child_node.getAttribute( 'alt' ) || '';
					if ( alt ) {
						parts.push( alt.trim() );
					}
					continue;
				}

				if ( ! child_node.hasAttribute( 'itemprop' ) ) {
					continue;
				}

				const inner_text = this.collectTextContent( child_node );
				if ( inner_text ) {
					parts.push( inner_text );
				}
			}
		} // for

		return parts.join( ' ' ).replace( /\s+/g, ' ' ).trim();
	}

	/**
	 * final_graph をJSON-LD形式 に変換
	 * 
	 * @param   {object} final_graph - 全エンティティのマップ
	 * @returns {array}
	 */
	buildJsonLdGraph( final_graph ) {
		let transformed_graph = [];

		for ( const item_id in final_graph ) {
			const entity = final_graph[ item_id ];
			let new_entity = {};

			if ( entity.id ) {
				new_entity[ '@id' ] = entity.id;
			}

			if ( entity.type ) {
				if ( entity.type.startsWith( 'https://schema.org/' ) ) {
					new_entity[ '@type' ] = entity.type.substring( 'https://schema.org/'.length );
				} else {
					new_entity[ '@type' ] = entity.type;
				}
			}

			if ( entity.properties ) {
				for ( const prop_key in entity.properties ) {
					const values = entity.properties[ prop_key ];
					let prop_values = [];

					values.forEach( item => {
						if ( undefined === item.value || null === item.value ) {
							return;
						}

						if ( item.is_reference ) {
							prop_values.push( { '@id': item.value } );
						} else {
							prop_values.push( item.value );
						}
					} );

					if ( 1 === prop_values.length ) {
						new_entity[ prop_key ] = prop_values[ 0 ];
					} else if ( 1 < prop_values.length ) {
						new_entity[ prop_key ] = prop_values;
					}
				} // for
			}

			transformed_graph.push( new_entity );
		} // for

		return transformed_graph;
	}

}
